;+
; NAME:
;  datamon
;
; PURPOSE:
;  Summary plot of raw data contained in photometry log file
;
; DESCRIPTION:
;  This program plots photometry data from files in the format written by
;    RDPHALT.  These files usually contain extracted photometry from a single
;    night's observations.  There are four plots generated by this program.
;
;    (1) Instrumental magnitude versus time.  This plots all objects and all
;        filters on the same plot.  For some types of data this plot can be
;        quite confusing.  For some datasets, this can show overall trends in
;        data quality.  Because of the scaling this plot usually only shows
;        the most glaring problems.
;
;    (2) Shifted instrumental magnitude versus time.  The first object and the
;        first filter is (arbitrarily) chosen as the "reference".  All other
;        object/filter combinations are shifted to have the same mean as the
;        reference.  The magnitude shown is of little importance but you can
;        see much more subtle trends in the data because the scaling doesn't
;        crunch things down as much.  Good data will show only variations
;        caused by airmass though these may not be readily discernable depending
;        on the data collection pattern.
;
;    (3) FWHM versus time.  All objects and all filters are plotted.  The FWHM
;        is the number directly from the file with no scaling or conversions.
;        Most of the time this value should be in arc seconds but there is
;        nothing present to allow the program to know.  It's up to you to know
;        what the absolute number means.  Superimposed on the plot are solid
;        curves, one for each filter that shows the trend for that filter over
;        time.  The curve is labeled with the name of the filter as read from
;        the log file.
;
;    (4) Sky signal versus time.  The sky signal from each observation is
;        plotted in photons/second.  The conversion to photons is done by using
;        the gain value listed in the log file.  Similar to the previous plot,
;        lines are superimposed to highlight the trends for each filter.
;
;  By default, the plots are sent to the current plotting device. This would
;    normally be the display.  When using the default plotting device, the
;    plots are sent to four windows (or pages).  Only the first plot is
;    identified by the file name being plotted.
;
;  If you use the /PRINT option, the plot is generated for one page and sent
;    to the default printer.  The contents of the plots are the same except
;    for being formatted to fit all on one page.
;
; CATEGORY:
;  Photometry
;
; CALLING SEQUENCE:
;  datamon,file
;
; INPUTS:
;  file - String of photometry log file to read and plot.
;
; OPTIONAL INPUT PARAMETERS:
;
; KEYWORD INPUT PARAMETERS:
;  PRINT - Flag, if set generates a plot to printer.
;  QUEUE - String, name of print queue to send plot to.  (overrides current
;             default printer, unix only).
;  NOQUEUE - Flag, if set suppresses sending the plot to the printer (is
;             only 
;
; OUTPUTS:
;
; KEYWORD OUTPUT PARAMETERS:
;
; COMMON BLOCKS:
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;
; MODIFICATION HISTORY:
;  Written by Marc W. Buie, Lowell Observatory, 2000/07/14.
;  2000/10/09, MWB, fixed sky plot bug, added lowess curves to sky
;  2001/04/18, MWB, changed systime call to use /julian flag
;-
pro datamon,file,PRINT=print,QUEUE=queue,NOQUEUE=noqueue

   if badpar(file,7,0,CALLER='DATAMON (file) ') then return
   if badpar(print,[0,1,2,3],0,CALLER='DATAMON (PRINT) ',default=0) then return
   if badpar(queue,[0,1,2,3],0,CALLER='DATAMON (QUEUE) ',default='') then return
   if badpar(noqueue,[0,1,2,3],0,CALLER='DATAMON (NOQUEUE) ',default=0) then return

   if not exists(file) then begin
      print,'File ',file,' not found.'
      return
   endif

   rdphalt,file,fn,obj,fil,jd,exptime,gain,rad,sky1,sky2,serial, $
           xpos,ypos,fwhm,maxcnt,sky,skyerr,mag,err,bad
   z = where(bad eq 0,count)
   if count eq 0 then begin
      print,'No good data in file ',file
      return
   endif
   if count ne n_elements(mag) then begin
      fn      = fn[z]
      obj     = obj[z]
      fil     = fil[z]
      jd      = jd[z]
      exptime = exptime[z]
      gain    = gain[z]
      rad     = rad[z]
      sky1    = sky1[z]
      sky2    = sky2[z]
      serial  = serial[z]
      xpos    = xpos[z]
      ypos    = ypos[z]
      fwhm    = fwhm[z]
      maxcnt  = maxcnt[z]
      sky     = sky[z]
      skyerr  = skyerr[z]
      mag     = mag[z]
      err     = err[z]
      bad     = bad[z]
   endif

   radr=minmax(rad)
   radstr = strn(radr[0],format='(f10.1)')
   if radr[0] ne radr[1] then $
      radstr = radstr + '-' + strn(radr[1],format='(f10.1)')

   sky1r=minmax(sky1)
   sky1str = strn(sky1r[0],format='(f10.1)')
   if sky1r[0] ne sky1r[1] then $
      sky1str = sky1str + '-' + strn(sky1r[1],format='(f10.1)')

   sky2r=minmax(sky2)
   sky2str = strn(sky2r[0],format='(f10.1)')
   if sky2r[0] ne sky2r[1] then $
      sky2str = sky2str + '-' + strn(sky2r[1],format='(f10.1)')

   skystr = '(' + sky1str + ',' + sky2str + ')'
   apstr = 'objrad ' + radstr + '  sky ' + skystr

   title = file + '  ' + apstr

   jd0 = long(jd[0]+0.5d0)-0.5d0

   if print then begin
      old_dev = !d.name
      p_font = !p.font
      if old_dev ne 'PS' or old_dev ne 'PRINTER' then set_plot,'PS'
      setpage,/landscape
      !p.font=0
   endif

   amag = mag

   fillist = fil[uniq(fil,sort(fil))]
   nfil    = n_elements(fillist)
   objlist = obj[uniq(obj,sort(obj))]
   nobj    = n_elements(objlist)

   time = (jd-jd0)*24.0
   psky = sky*gain/exptime

   magref = 0.0
   for i=0,nfil-1 do begin
      for j=0,nobj-1 do begin
         z=where(fil eq fillist[i] and obj eq objlist[j],count)
         if count gt 1 then begin
            if i eq 0 and j eq 0 then begin
               magref = mean(mag[z])
            endif else begin
               amag[z] = mag[z] - mean(mag[z]) + magref
            endelse
         endif else if count eq 1 then begin
            amag[z] = magref
         endif
      endfor
   endfor

   setusym,-1

   if print then !p.multi=[0,2,2]

   if not print then setwin,0
   ploterror,time,mag,err,psym=8,yr=maxmin(mag), $
      xtitle='Time (UT hours)',ytitle='Instrumental Magnitude', $
      title=title

   if not print then setwin,2
   ploterror,(jd-jd0)*24.0,amag,err,psym=8,yr=maxmin(amag), $
      xtitle='Time (UT hours)',ytitle='Shifted Instrumental Magnitude'

   setusym,1

   if not print then setwin,1
   plot,(jd-jd0)*24.0,fwhm,psym=5,xtitle='Time (UT hours)', $
      ytitle='FWHM'

   for i=0,nfil-1 do begin
      z=where(fil eq fillist[i],count)
      if count gt 1 then begin
         lowess,time[z],fwhm[z],1.0,fwhms
         oplot,time[z],fwhms
         xyouts,time[z[0]],fwhms[0],fillist[i],charsize=2.0,align=1.0
      endif
   endfor

   if not print then setwin,3
   plot,(jd-jd0)*24.0,psky,psym=5,xtitle='Time (UT hours)', $
      ytitle='Sky signal (photons/sec)'

   for i=0,nfil-1 do begin
      z=where(fil eq fillist[i],count)
      if count gt 1 then begin
         lowess,time[z],psky[z],0.25,pskys
         oplot,time[z],pskys
         xyouts,time[z[0]],pskys[0],fillist[i],charsize=2.0,align=1.0
      endif
   endfor

   if print then begin
      curjd   = systime(/julian,/utc)
      jdstr,curjd,10,str
      xyouts,1.00,0.00,str,align=1.0,charsize=0.5,/normal
      if not noqueue then hardcopy,QUEUE=queue,/silent
      if old_dev ne !d.name then set_plot,old_dev
      !p.font=p_font
      !p.multi=0
   endif

end
