;+
; NAME:
;  desalt
; PURPOSE:
;  Flag fake objects in .obj files that were originally generated by imsalt
; DESCRIPTION:
;  Read an object list from two files, one working obj file, one containing
;  the salted objects.  Compare them, determine which objects are salted and
;  write "s" into the working obj file to identify them as salted objects.
; CATEGORY:
;  Astrometry
; CALLING SEQUENCE:
;  rdobjlist,objfile,saltfile
; INPUTS:
;  objfiledir  - Location of working obj files to be read.
;  saltfiledir - Location of obj files containing just the salted objects.
; OPTIONAL INPUT PARAMETERS:
;
; KEYWORD INPUT PARAMETERS:
;  THRESHOLD - distance found-object can be from salted object (in pixels)
;                 default = 3 pixels
;
; OUTPUTS:
;
; KEYWORD OUTPUT PARAMETERS:
;
; COMMON BLOCKS:
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;
; MODIFICATION HISTORY:
;  2004/07/12, Written by Julie Kane, MIT
;  2004/07/18, JFK, added support for multiple files
;  2004/07/22, MWB, final debug
;-
pro desalt,objfiledir,saltfiledir,THRESHOLD=threshold

   self='DESALT: '
   if badpar(objfiledir,7,0,caller=self+'(objfiledir) ') then return
   if badpar(saltfiledir,7,0,caller=self+'(saltfiledir) ') then return
   if badpar(threshold,[0,2,3,4,5],0,caller=self+'(THRESHOLD) ', $
                                     default=3.0) then return

   objfiledir = addslash(objfiledir)
   saltfiledir = addslash(saltfiledir)

   if not exists(objfiledir) then begin
      print,self,'Working obj Dir ',objfiledir,' not found.'
      return
   endif

   if not exists(saltfiledir) then begin
      print,self,'Salt-obj Dir ',saltfiledir,' not found.'
      return
   endif

   pushd, objfiledir
   objfiles = file_search('*.obj', count=nfiles)
   popd

   if nfiles eq 0 then begin
      print, 'No .obj files in ',objfiledir
      return
   end

   mindist=1.0e9  ; minimum distance for non-matching objects

   for f=0,nfiles-1 do begin

      saltfile = saltfiledir + objfiles[f]
      objfile  = objfiledir  + objfiles[f]

      if not exists(saltfile) then begin
         if strmid(objfiles[f],0,1) eq 'f' or strmid(objfiles[f],0,1) eq 's' then $
            print,self,'Salt-obj File ',saltfile,' not found.  Skipping.'
         continue
      endif

      print,'objfile  = ',objfile,'    saltfile = ',saltfile

      dirty = 0
      rdoblist,saltfile,nobj2,filelist2,dt2,offset2,pos2,flags2,idstr2,nfiles2
      if nobj2 eq 0 then continue

      rdoblist,objfile, nobj1,filelist1,dt1,offset1,pos1,flags1,idstr1,nfiles1
      if nobj1 eq 0 then continue

      flagsf=flags1
      idstrf=idstr1

      z=where(flags1 eq 'y',count)
      if count gt 0 then begin
         ; i is the index of elements in the obj file that are not n's
         for i=0,count-1 do begin ; ie, the number of objects to be tested
            ; j is the index of objects in the salted file
            for j=0, nobj2-1 do begin

               pos=pos1[*,z[i]]
               sizepos = n_elements(pos)
               matches = 0

               ;count the number of positions that match and then process
               ; k is the num of coords that exists for the working object
               negcount=0
               for k=0, sizepos-2, 2 do begin
                  if pos2[k,j] lt 0.0 or pos[k] lt 0.0 then begin
                     negcount++
                     continue
                  endif
                  dist=sqrt( (pos2[k,j]-pos[k])^2 + (pos2[k+1,j]-pos[k+1])^2 )
                  if dist lt threshold then $
                     matches = matches + 1 $
                  else $
                     if dist lt mindist then mindist=dist
               endfor

               if matches eq (sizepos/2)-negcount then begin 
                  ;set flagsf[z[i]] = s and append salt to idstr[z[i]]
                  flagsf[z[i]] = 's'
                  dirty=1
               endif else if matches gt 0 then begin
                  ;set flagsf[z[i]] = n and append salt to idstr[z[i]]
                  flagsf[z[i]] = 'n'
                  if strlen(idstrf[z[i]]) gt 0 then $
                     idstrf[z[i]] = idstrf[z[i]]+',salt' $
                  else $
                     idstrf[z[i]] = 'salt'
                  print,'A non-salted object was partially matched', $
                        ' with a salted object'
                  dirty=1
               endif

            endfor ; loop over objects in second list

         endfor ; loop over yes objest in first list 

      endif

      if dirty then $
         wroblist,objfile,nobj1,filelist1,dt1,offset1,pos1,flagsf,idstrf,nfiles1

   endfor

   print,'Minimum non-matching distance was ',mindist,' pixels.'

end
