;+
; NAME:
;  psfgen
; PURPOSE:   (one line only)
;  Generate PSF files for a set of images
; DESCRIPTION:
;
; CATEGORY:
;  CCD data processing
; CALLING SEQUENCE:
;  psfgen,root
; INPUTS:
;  root       :String, the date when observations are taken (YYMMDD)
; OPTIONAL INPUT PARAMETERS:
; KEYWORD INPUT PARAMETERS:
;  IMDIR      :The path to the image data (default=current directory)
;  OVERWRITE  : Flag, if set will overwrite existing PSF files.  The default
;                is to skip processing images that already have a PSF.
;  PSFDIR     :The directory where the output PSF files are to be written.
;                The default is SRCDIR+'/psf'
;  SRCDIR     :The path to the source lists (default=current directory)
;                These files are normally generated by findsrc.
;  QUIET      :Flag, suppress all chatty printed output.
;  VIEWIT     ;Flag, set to look at the psf header (and subheader)
; KEYWORD OUTPUT PARAMETERS:
; COMMON BLOCKS:
; SIDE EFFECTS:
; RESTRICTIONS:
; PROCEDURE:
; MODIFICATION HISTORY:
;  2004/07/13, Written by Maureen Teyssier, REU Lowell Observatory
;  2004/07/16, MWB, incorporated into the master library
;  2004/08/13, MWB, fixed bug with psfdir non-default operation
;-

pro psfgen,root, $
       SRCDIR=srcdir,IMDIR=imdir,PSFDIR=psfdir,QUIET=quiet,VIEWIT=viewit, $
       OVERWRITE=overwrite

   self='PSFGEN: '
   if badpar(root,7,0,CALLER=self+'(root) ') then return

   if badpar(srcdir,[0,7],0,CALLER=self+'(SRCDIR) ',DEF='') then return
   if badpar(imdir,[0,7],0,CALLER=self+'(IMDIR) ',DEF='') then return
   if badpar(quiet,[0,1,2,3],0,CALLER=self+'(QUIET) ',DEF=0) then return
   if badpar(viewit,[0,1,2,3],0,CALLER=self+'(VIEWIT) ',DEF=0) then return
   if badpar(overwrite,[0,1,2,3],0,CALLER=self+'(OVERWRITE) ',DEF=0) then return

   srcdir=addslash(srcdir)
   imdir=addslash(imdir)

   if badpar(psfdir,[0,7],0,CALLER=self+'(PSFDIR) ',DEF='') then return

   if psfdir eq '' then $
      psfdir=addslash(srcdir+'psf') $
   else $
      psfdir=addslash(psfdir)

   if not exists(psfdir) then file_mkdir,psfdir,/noexpand_path

   if not exists(imdir) then begin
      print,self,'The image directory ',imdir,' does not exist.  Aborting.'
      return
   endif

   if not exists(srcdir) then begin
      print,self,'The src file directory ',srcdir,' does not exist.  Aborting.'
      return
   endif

   if not exists(psfdir) then begin
      print,self,'The psf directory ',psfdir,' does not exist.  Aborting.'
      return
   endif

   ;.src and .fits files
   pushd,imdir
   imlist=file_search(root+'.*',count=nfiles)
   popd

   if nfiles eq 0 then begin
      print,self,'No files found in ',imdir,'.  Quitting.'
      return
   endif

   for ifile=0,nfiles-1 do begin ; file loop

      if !version.os_family eq 'unix' then begin
         if file_test(imdir+imlist[ifile],/dangling_symlink) then begin
            print,self,'Warning: ',imdir+imlist[ifile], $
                       ' is a broken symbolic link.  Skipping.'
            continue
         endif
      endif

      imfile = imlist[ifile]
      tag = strmid(imfile,4,99,/reverse_offset)
      if tag eq '.fits' then begin
         imfile = strmid(imfile,0,strlen(imfile)-5)
         ft = '.fits'
      endif else begin
         ft = ''
      endelse

      psfname = imfile+'.psf'

      if not quiet then print,'PSF:',psfdir+psfname
 
      ;check to see if the psf exists, and skip it if it does
      if exists(psfdir+psfname) then begin
         if not quiet then print,'Skipping, already exists.'
         continue
      endif
      
      ; Check header of image to see if it is a multi-extension image,
      ;  and build an extension list if it is
      hdr=headfits(imdir+imfile+ft) 
      numext=sxpar(hdr,'NEXTEND')
      extend=sxpar(hdr,'EXTEND')

      if numext eq 0 or extend ne 1 then extlist=0 else extlist=indgen(numext)+1
      numext=n_elements(extlist) 

      for ext=0, numext-1 do begin;extension loop

         if extlist[ext] eq 0 then begin
            if not quiet then print,'Reading ',imdir+imfile+ft
            image=readfits(imdir+imfile+ft,hdr)
            exttag='.src'
         endif else begin
            if not quiet then print,'Reading ',imdir+imfile+ft, $
                                    ' extension ',strn(extlist[ext])
            image=readfits(imdir+imfile+ft,s_hdr,exten_no=extlist[ext])
            extstr=strb36(extlist[ext])
            exttag='.srcx'+extstr
         endelse 

         srcname=imfile+exttag
         if not quiet then print,'Reading ',srcdir+srcname
         if exists(srcdir+srcname) then begin;SRC LOOP

            src=readfits(srcdir+srcname,srchdr)

            objrad=   sxpar(srchdr,'OBJRAD')
            srcgain=  sxpar(srchdr,'GAIN')
            exptime=  sxpar(srchdr,'EXPTIME')
            skylevel= sxpar(srchdr,'SKYLEVEL')

            srcx    = src[*,0]
            srcy    = src[*,1]
            srcfwhm = src[*,2]

            ;-------generating a better psf
            ;find the mean FWHM and choosing obj. w/in 2 sigma
            bad=bytarr(n_elements(srcx))
            robomean,srcfwhm,2.0,0.5,bad=bad
            z=where(bad eq 0,nobjs)
            srcx=srcx[z]
            srcy=srcy[z]
            if nobjs gt 200 then begin
               idx=long(randomu(seed,200)*(nobjs-1)+0.5)
               nbos = 200
               srcx=srcx[idx]
               srcy=srcy[idx]
            endif

            ; this will set the size of the psf image array
            dw=3*fix(objrad+0.5)

            if not quiet then print, 'PSF GENERATED: '

            ; generate psf
            image=float(image)
            psfstack,image,srcx,srcy,psf,dw=dw

            ; integrate the psf over a circle of radius=objrad and determine
            ;   the centroid position at the same time.  The integral over the
            ;   entire psf that was generated is normalized to unity.
            basphote,1.0,psf,1.0,dw+1,dw+1,objrad,0.,-1., $
                 flux=flux,fwhm=fwhm,xcen=xcen,ycen=ycen,max=max,/silent,/nolog

            ;-------writing the psf file
            if extlist[ext] eq 0 then begin ; no extension case

               mkhdr,psfhdr,psf
               sxaddpar,psfhdr,'XCEN',xcen,' centroided position of psf'
               sxaddpar,psfhdr,'YCEN',ycen,' centroided position of psf'
               sxaddpar,psfhdr,'OBJRAD',objrad,' Object radius'
               sxaddpar,psfhdr,'FWHM',fwhm,' FWHM in pixels of PSF'
               sxaddpar,psfhdr,'FLUX',flux, $
                               ' Fraction of flux contined within OBJRAD'
               sxaddpar,psfhdr,'MAX',max,' Peak pixel value in PSF'
               writefits,psfdir+psfname,psf,psfhdr

            endif else begin ; multiextension case

               ; take care of primary header for group fits file
               if extlist[ext] eq 1 then begin
                  mkhdr,psfhdr,'',/extend
                  sxaddpar,psfhdr,'NEXTEND',numext, $
                     ' Number of image extensions',after='extend'
                  sxaddpar,psfhdr,'OBJRAD',objrad,' Object radius'
                  if viewit then begin
                     zf = 4
                     setwin,0,xsize=zf*(2*dw+1),ysize=zf*(2*dw+1)
                     tvscl,rebin(psf,zf*(2*dw+1),zf*(2*dw+1),/sample)
                  endif
                  writefits,psfdir+psfname,0,psfhdr
               endif 

               mkhdr,psfs_hdr,psf,/image
               sxaddpar,psfs_hdr,'XCEN',xcen,' centroided position of psf'
               sxaddpar,psfs_hdr,'YCEN',ycen,' centroided position of psf'
               sxaddpar,psfs_hdr,'FWHM',fwhm,' FWHM in pixels of PSF'
               sxaddpar,psfs_hdr,'FLUX',flux, $
                                 ' Fraction of flux contined within OBJRAD'
               sxaddpar,psfs_hdr,'MAX',max,' Peak pixel value in PSF'
               if viewit then begin
                  zf = 4
                  setwin,0,xsize=zf*(2*dw+1),ysize=zf*(2*dw+1)
                  tvscl,rebin(psf,zf*(2*dw+1),zf*(2*dw+1),/sample)
                  print,'(x,y)',xcen-dw,ycen-dw, $
                        '  fwhm,flux,max',fwhm,flux,max, $
                        format='(a,2(1x,f5.2),a,1x,f4.1,1x,f5.3,1x,f6.4)'
               endif

               writefits,psfdir+psfname,psf,psfs_hdr,/append

            endelse ;writing file

         endif else begin; SRC LOOP
            print,self,'No src file for ',imfile
         endelse

      endfor ;ext

   endfor;ifile

end
