;+
; NAME:
;  starproc
;
; PURPOSE:
;  Collect and process final (standard) star photometry
;
; DESCRIPTION:
;  This program processes the accumulated stellar photometry into final
;    averages.  The data are generated by REDUCTOR and have been saved
;    to the final output directory (set via PATH keyword).
;    This program can reduce all stars in this directory or just one star.
;
;  On starting the program, the contents of the photometry directory are
;    scanned and a selection widget is presented.  You can either choose
;    a single object by its name or choose the item "ALL".  If you do one
;    object, it will be analyzed and plots of the data are provided.  If
;    you choose ALL, each star in turn is analyzed but no plots are generated.
;
;  The photometry for each star is examined and a mean for each color is
;    found.  The mean comes from weighted average of all points.  After the
;    mean is determined, the photometry is examined against the mean.  Any
;    point found to be greater than 5 sigma from the mean is discarded and
;    the mean is recalculated.  If plotted, these removed points will plot
;    with a different symbol (filled circle with overlayed asterisk).  This
;    point removal continues until no points are removed or 10 passes are
;    made.
;
;  The final uncertainty for the combined photometry is comes from taking the
;    standard deviation of the data and dividing by the sqrt of the number
;    of nights of data (NOT the number of points).  My assumption is that
;    the errors tend to be systematic within a night but random from night
;    to night.  Obviously, the photometry will not be particularly trustworthy
;    until there are quite a few nights in the average.
;
;  The final output file is a listing, one line per star, of the final averaged
;    values.  Left to right, the values are:
;  Name, B (mag, err, npoints, nights, chisq),
;        V (mag, err, npoints, nights, chisq),
;        R (mag, err, npoints, nights, chisq),
;        B-V (color, err)
;        V-R (color, err)
;
; CATEGORY:
;  Photometry
;
; CALLING SEQUENCE:
;  starproc[,PATH=path]
;
; INPUTS:
;  This is an interactive program, there are no explicit inputs.
;
; OPTIONAL INPUT PARAMETERS:
;
; KEYWORD INPUT PARAMETERS:
;  PATH - Directory where the star data is located.  Default is
;           /frakir/raid/buie/photometry/stars
;
; OUTPUTS:
;  output is written to a file ALL.DAT in the stars directory.
;
; KEYWORD OUTPUT PARAMETERS:
;
; COMMON BLOCKS:
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;
; MODIFICATION HISTORY:
;  97/02/27, Written by Marc W. Buie, Lowell Observatory
;  2004/02/09, MWB, changed default for PATH
;-
PRO starproc,PATH=path

   IF badpar(path,[0,7],0,CALLER='STARPROC: (PATH) ', $
         default='/frakir/raid/buie/photometry/stars') THEN return

   blanks='                   '

   ; Go to the data storage area
   cd,path,current=old_dir

   spawn,'ls',files

   period=strpos(files,'.')
   root=files
   FOR i=0,n_elements(files)-1 DO BEGIN
      root[i]=strmid(root[i],0,period[i])
   ENDFOR
   starnames=root[uniq(root,sort(root))]

   todo = picker(starnames,ysize=30)
   IF todo eq '[[[CANCEL]]]' THEN return

   IF todo eq 'ALL' THEN todo = starnames[where(starnames ne 'ALL')]
   numtodo = n_elements(todo)

   cleanname = repchar(todo,'_',' ')

   filters=['B','V','R']
   nfilters=n_elements(filters)

   !p.multi=[0,1,2]
   FOR j=0,numtodo-1 DO BEGIN
      found=intarr(nfilters)
      mags=fltarr(nfilters)
      errs=fltarr(nfilters)
      magstr=strarr(nfilters)
      if numtodo eq 1 then print,todo[j] $
      else print,todo[j]+blanks,format='($,a21,1x)'
      FOR i=0,nfilters-1 DO BEGIN
         fn = todo[j]+'.'+filters[i]
         IF exists(fn) THEN BEGIN
            IF numtodo gt 1 THEN print,filters[i],format='($,1x,a)'
            readcol,fn,date,colred,mag,err,color,colorerr, $
               format='i,a,f,f,f,f',/silent
            nights=date[uniq(date,sort(date))]
            numni=n_elements(nights)
            npoints=n_elements(mag)
            bad=intarr(npoints)
            zg=where(bad eq 0,ngood)

            pass=0
            REPEAT BEGIN
               meanerr,mag[zg],err[zg],avg,sigmam,sigmad
               dev=(mag-avg)/err
               chisq=total(dev[zg]^2)/ngood
               oldgood = ngood
               bad = dev gt 5.0
               zg=where(bad eq 0,ngood)
               IF numtodo eq 1 THEN BEGIN
                  print,pass,filters[i],avg,sigmam,sigmad,sigmad/sqrt(numni), $
                     npoints,n_elements(nights),chisq, $
                     format='(i1,1x,a,1x,f7.4,1x,f6.4,1x,f6.4,1x,f6.4,1x,i3,1x,i2,1x,f5.2)'
               ENDIF ELSE BEGIN
                  print,pass,format='($,i1)'
               ENDELSE
               pass=pass+1
            ENDREP UNTIL ngood eq oldgood or pass eq 10


            IF numtodo eq 1 THEN BEGIN
               setwin,i+1
               ploterror,indgen(npoints),mag,err,yr=maxmin(mag),psym=8, $
                  ytitle=filters[i]+' Magnitude',title=todo[j]
               plot,dev,psym=8
               zb=where(bad eq 1,nbad)
               IF nbad ne 0 THEN BEGIN
                  oplot,zb,dev[zb],psym=2,symsize=1.5
               ENDIF
;               stats,mag,/silent,nbins=50
            ENDIF
            found[i]=1
            mags[i]=avg
            errs[i]=sigmad/sqrt(numni)
            magstr[i]=string(mags[i],errs[i],ngood,n_elements(nights),chisq, $
                                format='(f7.4,1x,f6.4,1x,i3,1x,i2,1x,f5.2)')
         ENDIF
      ENDFOR ; Filter loop on I

      cfound = found[0:1] and found[1:2]
      ncolors=n_elements(cfound)
      colstr=strarr(ncolors)
      colors=mags[0:1]-mags[1:2]
      colorerr=sqrt(errs[0:1]^2+errs[1:2]^2)
      FOR i=0,n_elements(cfound)-1 DO BEGIN
         IF cfound[i] THEN BEGIN
            colstr[i]=string(colors[i],colorerr[i],format='(f7.4,1x,f6.4)')
         ENDIF
      ENDFOR

      tag=strmid(todo[j]+blanks,0,21)
      info=string(magstr,colstr,format='(3(2x,a27),2(2x,a14))')
      repwrite,'ALL.DAT',tag,tag+info
      IF numtodo gt 1 THEN print,''

      ; Update this star in the standard output file.
      IF exists('../buiestd.dat') THEN BEGIN
         rdphocat,'../buiestd.dat',bname,bmags,bcodes,bfilnam,bnfil
         z = where(cleanname[j] eq bname)
         IF z[0] eq -1 THEN BEGIN
            bname = [bname,cleanname[j]]
            bmags = [[bmags],[mags]]
            bcodes= [[bcodes],[found]]
         ENDIF ELSE BEGIN
            bname[z] = cleanname[j]
            bmags[*,z] = mags
            bcodes[*,z]= found
         ENDELSE
      ENDIF ELSE BEGIN
         bname = cleanname[j]
         bmags = mags
         bcodes= found
         bfilnam= filters
      ENDELSE
      idx=sort(bname)

      wrphocat,'../buiestd.dat',bname[idx],bmags[*,idx],bcodes[*,idx],bfilnam

   ENDFOR

   ; Restore working directory
   cd,old_dir
   !p.multi=0
END
