; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
;+
; NAME:
;     DIALOG_PLOT
;
; PURPOSE:
;     A modal (blocking) dialog widget to display a plot.  
;     The dialog must be dismissed with the 'Dismiss' button before
;     execution of the calling program can continue.
;
; TYPE:
;     FUNCTION
;
; CATEGORY:
;     WIDGETS
;
; CALLING SEQUENCE:
;     result = DIALOG_PLOT (XDATA = xData, YDATA = yData)
;
; INPUTS:
;     XDATA and YDATA (via keyword, see below)
;
; KEYWORD PARAMETERS:
;
;     XDATA:         Array of independent data.
;     YDATA:         Array of dependent data.  
;
;     EXECUTE:       STRARR containing one or more IDL statements to execute. 
;                    The first command in the string must be a PLOT command.  
;                    The parameters xData, yData are available for use in the 
;                    EXECUTE string. See below for an example.
;  
;     PS_OPTION:     Add optional button to create a Postscript file
;     GIF_OPTION:    Add optional button to create a gif image
;
;     WINTITLE:      Window title [default = 'dialog_plot']
;
;     XSIZE:         The width of the plot region  (pixels) [default = 500]
;     YSIZE:         The height of the plot region (pixels) [default = 400]
;
;     DIALOG_PARENT: Set this keyword to the widget ID of a widget over
;                    which the message dialog should be positioned. When 
;                    displayed, the DIALOG_PLOT dialog will be positioned over 
;                    the specified widget. Dialogs are often related to a 
;                    non-dialog widget tree.  The ID of the widget in that 
;                    tree to which the dialog is most closely related should 
;                    be specified.
;
;     All PLOT keywords also accepted.
;
; OUTPUTS:
;     result: 1, or 0 for failure (missing parameters, or EXECUTE failed)
;
; COMMON BLOCKS:
;     NONE
;
; SIDE EFFECTS:
;     Creates a modal widget
;
; RESTRICTIONS:
;     NONE
;
; DEPENDENCIES:
;     NONE
;
; EXAMPLE:
;
;     xData = [0, 100]
;     yData = [-100, 100]
;
;     ; Create a simple plot, with an option to print to Postscript file.
;     ;   
;     result = DIALOG_PLOT (XDATA = xData, YDATA = yData, $
;         winTitle = 'Demo', /PS_OPTION)
;
;     ; Create a custom plot, with an option to print to gif file.  Note 
;     ; that EXECUTE can utilize the XDATA, YDATA keywords.
;     ;   
;     result = DIALOG_PLOT (XDATA = xData, YDATA = yData, EXECUTE = [ $
;         'PLOT, /NODATA, XDATA, YDATA', $
;         'OPLOT, [50, 50], !Y.CRANGE', $
;         'OPLOT, [70, 70], !Y.CRANGE, LINESTYLE = 1' ], $
;         XTITLE = 'xTitle', YTITLE = 'yTitle', winTitle = 'Demo', /GIF_OPTION)
;
; MODIFICATION HISTORY:
;     v0.1: written, Robert.Mallozzi@msfc.nasa.gov, March 1998.
;
;-
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 

FUNCTION DIALOG_PLOT, $
    XDATA = xData, YDATA = yData, $
    EXECUTE = execute_s, $
    WINTITLE = winTitle, $
    XSIZE = xSize, YSIZE = ySize, $
    PS_OPTION = ps_option, GIF_OPTION = gif_option, $
    DIALOG_PARENT = dialog_parent, $
    _EXTRA = EXTRA


haveXdata = (N_ELEMENTS(xData) NE 0)
haveYdata = (N_ELEMENTS(yData) NE 0)

doExecute = (N_ELEMENTS(execute_s) NE 0)

IF (NOT haveXdata) OR (NOT haveYdata) THEN BEGIN
   MESSAGE, /INFO, 'Usage: result = DIALOG_PLOT (XDATA = xData, YDATA = yData)'
   RETURN, 0
ENDIF

pSave = !P
xSave = !X
ySave = !Y

IF (N_ELEMENTS(winTitle) EQ 0) THEN $
   winTitle = 'dialog_plot'

HAVE_PARENT = N_ELEMENTS(dialog_parent) NE 0

usePsOption  = N_ELEMENTS(ps_option) NE 0
useGifOption = N_ELEMENTS(gif_option) NE 0

IF (N_ELEMENTS(XSIZE) EQ 0) THEN BEGIN
   xSize = 500
ENDIF ELSE BEGIN
   xSize = XSIZE
ENDELSE

IF (N_ELEMENTS(YSIZE) EQ 0) THEN BEGIN
   ySize = 400
ENDIF ELSE BEGIN
   ySize = YSIZE
ENDELSE

; Top level base
;
IF (HAVE_PARENT) THEN BEGIN

   ; Check for a valid widget id
   ;
   HAVE_PARENT = WIDGET_INFO (LONG(dialog_parent), /VALID_ID)

ENDIF   

IF (HAVE_PARENT) THEN BEGIN
   base = WIDGET_BASE (TITLE = winTitle, /COLUMN, $
       /FLOATING, /MODAL, GROUP_LEADER = dialog_parent)
ENDIF ELSE BEGIN
   base = WIDGET_BASE (TITLE = winTitle, /COLUMN, MAP = 0)
ENDELSE

; Plotting canvas
;
winSave = !D.WINDOW
w = WIDGET_DRAW (base, XSIZE = xsize, YSIZE = ysize, /FRAME)
WIDGET_CONTROL, w, GET_VALUE = win
WSET, win

; Option buttons
;
IF (usePsOption) THEN $
   w = WIDGET_BUTTON (base, VALUE = 'Save to Postscript File')

IF (useGifOption) THEN $
   w = WIDGET_BUTTON (base, VALUE = 'Save to gif File')

w = WIDGET_BUTTON (base, VALUE = 'Dismiss')


; Map to screen
;
WIDGET_CONTROL, base, /REALIZE


; Place the dialog: window manager dependent
;
IF (NOT HAVE_PARENT) THEN BEGIN

   CURRENT_SCREEN = GET_SCREEN_SIZE()
   WIDGET_CONTROL, base, TLB_GET_SIZE = DIALOG_SIZE

   DIALOG_PT = [(CURRENT_SCREEN[0] / 2.0) - (DIALOG_SIZE[0] / 2.0), $ 
                (CURRENT_SCREEN[1] / 2.0) - (DIALOG_SIZE[1] / 2.0)] 

   WIDGET_CONTROL, base, $
                   TLB_SET_XOFFSET = DIALOG_PT[0], $
                   TLB_SET_YOFFSET = DIALOG_PT[1]
   WIDGET_CONTROL, base, MAP = 1

ENDIF

; Plot data
;
status = 1
IF (doExecute) THEN BEGIN

   ; Build the scalar command required by IDL's EXECUTE function
   ;
   command = ''
   
   ; Pass along any additional keywords to the PLOT call
   ; TODO: verify that first command calls PLOT
   ;
   command = command + execute_s[0] + ', _EXTRA = EXTRA' 

   FOR i = 1, N_ELEMENTS(execute_s) - 1 DO $
       command = command + ' & ' + execute_s[i]

   status = EXECUTE (command)
   IF (NOT status) THEN BEGIN
      MESSAGE, /INFO, 'EXECUTE failed: ' + command
      GOTO, ERROR
   ENDIF
   
ENDIF ELSE BEGIN

   PLOT, xData, yData, _EXTRA = EXTRA

ENDELSE

; Get the event, without using XMANAGER
;
NEXT:
event = WIDGET_EVENT (base)
WIDGET_CONTROL, event.id, GET_VALUE = value

IF (value EQ 'Save to Postscript File') THEN BEGIN

   fileName = DIALOG_PICKFILE (/WRITE, FILTER = '*.ps', $
       DIALOG_PARENT = event.top)
   IF (fileName NE '') THEN BEGIN

      d = FINDFILE (fileName, COUNT = fileExists)
      IF (fileExists) THEN BEGIN
         IF (DIALOG_MESSAGE (/QUESTION, ['File ' + fileName + ' exists.', $
	     'Overwrite?'], DIALOG_PARENT = event.top) EQ 'No') THEN $
            GOTO, NEXT
      ENDIF

      deviceSave = !D.NAME
      SET_PLOT, 'PS'
      DEVICE, FILE = fileName
      PLOT, xData, yData, _EXTRA = EXTRA
      SET_PLOT, deviceSave

   ENDIF

   GOTO, NEXT
   
ENDIF

IF (value EQ 'Save to gif File') THEN BEGIN

   fileName = DIALOG_PICKFILE (/WRITE, FILTER = '*.gif', $
       DIALOG_PARENT = event.top)
   IF (fileName NE '') THEN BEGIN

      d = FINDFILE (fileName, COUNT = fileExists)
      IF (fileExists) THEN BEGIN
         IF (DIALOG_MESSAGE (/QUESTION, ['File ' + fileName + ' exists.', $
	     'Overwrite?'], DIALOG_PARENT = event.top) EQ 'No') THEN $
            GOTO, NEXT
      ENDIF

      WRITE_GIF, fileName, TVRD()      
      
   ENDIF

   GOTO, NEXT
   
ENDIF


ERROR:

!P = pSave
!X = xSave
!Y = ySave

WSET, winSave


WIDGET_CONTROL, base, /DESTROY
RETURN, status



END
