; ----------------------------------------------------------------------------
;+
; NAME:
;     DRAW_ELLIPSE (procedure)
;
; PURPOSE:
;     Draw an ellipse in the x-y plane, centered at (cx, cy) of major 
;     radius rx, minor radius ry.
;
; CALLING SEQUENCE:
;     DRAW_ELLIPSE, cx, cy, rx, ry
;
; INPUTS:
;     cx : x center point of ellipse
;     cy : y center point of ellipse
;     rx : x-axis radius of ellipse 
;         (parallel to x-axis, use ROTATE keyword to rotate) 
;     ry : y-axis radius of ellipse 
;         (parallel to y-axis, use ROTATE keyword to rotate) 
;
; KEYWORDS:
;
;     COLOR : Color index (D = !P.COLOR) 
;
;     FILL : Draw a filled polygon; if this keyword is set, any valid POLYFILL
;         keyword may also be used (D = no fill)
;
;         For non-filled polygons, any valid oplot keywords are accepted 
;         (LINESTYLE, PSYM, etc).
;
;     RESOLUTION : INTEGER specifying number of points used to draw the
;         ellipse (D = 32)
;
;     ROTATE : Set the rotation in degees of the ellipse; by default, the major
;         axis is parallel to the x-axis (horizontal), and the 
;         minor axis is parallel to the y-axis (vertical).  Set this
;         keyword to an INTEGER specifying the number of degrees
;         to rotate the ellipse in a counter-clockwise direction.
;
;
; COMMON BLOCKS:
;     NONE
;
; SIDE EFFECTS:
;     None known
;
; RESTRICTIONS:
;     None known
;
; DEPENDENCIES:
;     NONE
;
; EXAMPLE:
;
;     ; Establish coordinate system
;     ;
;     PLOT, INDGEN (10), /NODATA
;
;     ; Load a color table
;     ;
;     LOADCT, 5
;
;     ; Plot an ellipse
;     ;
;     DRAW_ELLIPSE, 6, 2, 0.5, 1, /FILL, COLOR = 100
;
;     ; Outline it in a different color
;     ;
;     DRAW_ELLIPSE, 6, 2, 0.5, 1, THICK = 4, COLOR = 30
;     
;     ; Plot a filled circle; use POLYFILL keywords to draw hash marks
;     ;
;     DRAW_ELLIPSE, 6, 6, 1, 1, /FILL, /LINE_FILL, ORIENTATION = 45
;
; MODIFICATION HISTORY:
;     Written, 1999 August, Robert.Mallozzi@msfc.nasa.gov
;
;-
; ----------------------------------------------------------------------------

PRO DRAW_ELLIPSE, cx, cy, rx, ry, COLOR = color, RESOLUTION = resolution, $
    ROTATE = rotate, FILL = fill, _EXTRA = extra


    IF (N_ELEMENTS (color) EQ 0) THEN $
       color = !P.COLOR

    IF (N_ELEMENTS (resolution) EQ 0) THEN $
       resolution = 32

    IF (N_ELEMENTS (rotate) EQ 0) THEN $
       rotate = 0

    a = DINDGEN (resolution) * (!DPI * 2 / (resolution - 1))
     
    cosa = COS (a)
    sina = SIN (a)
     
    IF (rotate NE 0) THEN BEGIN
    
       rotate = rotate * !DTOR
       cosr = COS (rotate)
       sinr = SIN (rotate)

       x = rx * cosa
       y = ry * sina

       xp = cx + x * cosr - y * sinr  
       yp = cy + x * sinr + y * cosr   

    ENDIF ELSE BEGIN
    
       xp = cx + rx * cosa
       yp = cy + ry * sina
       
    ENDELSE

    PLOTS, xp, yp, COLOR = color, _EXTRA = extra
    
    IF (KEYWORD_SET (fill)) THEN $
       POLYFILL, xp, yp, COLOR = color, _EXTRA = extra
           
END
