; Copyright (c) 1999, Forschungszentrum Juelich GmbH ICG-3
; All rights reserved.
; Unauthorized reproduction prohibited.
;
;+
; NAME:
;	w_mean
;
; PURPOSE:
;	This function calculates a weighted mean and its error.
;
; CATEGORY:
;	MATH
;
; CALLING SEQUENCE:
;	result = w_mean(data [,weights])
;
; INPUTS:
;	data:	The data set for which the mean will be calculated.
;
; OPTIONAL INPUTS:
;	weights:	Weights to be used to calculate the weighted mean (see procedure
;		below). This array must have the same number of elements as data.
;		If not supplied the weights are assumed to be one.
;		Usually weights are choosen as: weights = 1 / s^2
;		where s defines the error of data.
;
; KEYWORD PARAMETERS:
;	STERR = sterr: If set to a named variable the error of the weighted mean is returned.
;	/DATAERR: If set the optional parameter weights is interpreted as vector of data errors.
;		The weights are then calculated as 1 / s^2
;	/NAN: Set this keyword to cause the routine to check for occurrences of the
;		IEEE floating-point value NaN in the input data. Elements with the value NaN
;		are treated as missing data.
;
; OUTPUTS:
;	This function returns the weighted mean of the input array.
;
; PROCEDURE:
;
;	wmean = TOTAL(weights * data) / TOTAL(weights)
;
;	exf = SQRT( TOTAL(weights * (data - wmean)^2) / ( (N - 1) * TOTAL(weights) ) )
;
;	inf = 1 / SQRT(TOTAL(weights))
;
;	All calculations are performed in double precision
;
; EXAMPLE:
;	data = [1., 2., 3., 4., 5.]
;	weights = [0.7, 0.4, 0.2, 0.9, 1.3]
;	PRINT, w_mean(data, weights, STERR = sterr), sterr
;	--->	3.4857143      0.77801422
;
; MODIFICATION HISTORY:
; 	Written by:	Frank Holland, 05.08.1999
;	20.07.2005: Added keyword parameters EXTERNAL_ERROR and INTERNAL_ERROR and DATAERR
;			The keyword parameter STERR returns now the larger of exF and inF
;-

FUNCTION w_mean, data, weights, STERR = sterr, EXTERNAL_ERROR = exf, INTERNAL_ERROR = inf, NAN = nan, DATAERR = dataerr

	ON_ERROR, 2
	IF N_PARAMS() EQ 0 THEN MESSAGE, 'Incorrect number of arguments.'

	ndata = N_ELEMENTS(data)
	IF N_PARAMS() EQ 1 THEN weights = REPLICATE(1.D, ndata)
	nwght = N_ELEMENTS(weights)
	IF ndata NE nwght THEN MESSAGE, 'Dimensions of data and weights do not agree.'

	IF KEYWORD_SET(nan) THEN BEGIN
		find = WHERE(FINITE(data) EQ 1, nfind)
		IF nfind GT 0 THEN $
			RETURN, w_mean(data[find], weights[find], STERR = sterr, EXTERNAL_ERROR = exF, INTERNAL_ERROR = inF) $
				ELSE RETURN, !VALUES.F_NAN
	ENDIF

	IF KEYWORD_SET(dataerr) THEN weights2 = 1D / weights^2 ELSE weights2 = weights
	wmean = TOTAL(data * weights2, /DOUBLE) / TOTAL(weights2, /DOUBLE)
	var = TOTAL(weights2 * (data - wmean)^2, /DOUBLE) / ( (ndata - 1) * TOTAL(weights2, /DOUBLE) )
	exf = SQRT(var)
	inf = 1D / SQRT(TOTAL(weights2, /DOUBLE))
	sterr = exf > inf

	RETURN, wmean
END
